const fs = require('fs');
const path = require('path');

function createDirectoryWatcher({ chokidar, videoExtensions, scanDirectory, getMainWindow }) {
  const watchers = new Map();

  function stopAllWatchers() {
    watchers.forEach((watcher) => watcher.close());
    watchers.clear();
  }

  function startWatchingDirectories(directories) {
    stopAllWatchers();

    directories.forEach((dirObj) => {
      const dirPath = typeof dirObj === 'string' ? dirObj : dirObj.path;
      if (!dirPath || !fs.existsSync(dirPath)) {
        return;
      }

      const watcher = chokidar.watch(dirPath, {
        ignored: /(^|[\/\\])\../,
        persistent: true,
        ignoreInitial: false,
        depth: 99
      });

      watcher.on('add', (filePath) => {
        const ext = path.extname(filePath).toLowerCase().slice(1);
        if (videoExtensions.includes(ext)) {
          const mainWindow = getMainWindow();
          if (mainWindow) {
            mainWindow.webContents.send('video-added', filePath);
          }
        }
      });

      watcher.on('unlink', (filePath) => {
        const ext = path.extname(filePath).toLowerCase().slice(1);
        if (videoExtensions.includes(ext)) {
          const mainWindow = getMainWindow();
          if (mainWindow) {
            mainWindow.webContents.send('video-removed', filePath);
          }
        }
      });

      watchers.set(dirPath, watcher);
    });

    directories.forEach((dirObj) => {
      const dirPath = typeof dirObj === 'string' ? dirObj : dirObj.path;
      if (!dirPath || !fs.existsSync(dirPath)) {
        return;
      }
      const videos = scanDirectory(dirPath);
      const mainWindow = getMainWindow();
      if (mainWindow && videos.length > 0) {
        const dirName = typeof dirObj === 'string' ? path.basename(dirPath) : dirObj.name;
        mainWindow.webContents.send('videos-scanned', {
          directory: dirPath,
          directoryName: dirName,
          videos
        });
      }
    });
  }

  return {
    startWatchingDirectories,
    stopAllWatchers
  };
}

module.exports = {
  createDirectoryWatcher
};

